# Check for 2 input files
if ($args.Count -ne 2) {
    Write-Host "Usage: Drag and drop two files onto this script."
    pause
    exit
}

$file1 = $args[0]
$file2 = $args[1]

# Ensure both files are from the same folder
$folder1 = Split-Path -Parent $file1
$folder2 = Split-Path -Parent $file2
if ($folder1 -ne $folder2) {
    Write-Host "Both files must be in the same folder."
    pause
    exit
}

# Get folder name and file names
$folderName = Split-Path $folder1 -Leaf
$name1 = [System.IO.Path]::GetFileNameWithoutExtension($file1)
$name2 = [System.IO.Path]::GetFileNameWithoutExtension($file2)

# Set output path
$outputName = "merged-$folderName" + "_" + "$name1" + "_" + "$name2"
$outputPath = Join-Path $folder1 $outputName


#Merge Params
$chunkSize = 0x400000  # 4MB
$maxSize = 0x800000    # 8MB

function Read-Chunk($filePath, $chunkSize) {
    $buffer = New-Object byte[] $chunkSize
    $bytesRead = 0
    $fs = [System.IO.File]::OpenRead($filePath)
    try {
        $bytesRead = $fs.Read($buffer, 0, $chunkSize)
    } finally {
        $fs.Close()
    }

    if ($bytesRead -lt $chunkSize) {
        for ($i = $bytesRead; $i -lt $chunkSize; $i++) {
            $buffer[$i] = 0xFF
        }
    }

    return ,$buffer
}

# Read chunks
$data1 = Read-Chunk $file1 $chunkSize
$data2 = Read-Chunk $file2 $chunkSize

# Merge
[byte[]]$merged = $data1 + $data2

if ($merged.Length -ne $maxSize) {
    Write-Host "Error: merged size is not maxSize"
    pause
    exit
}

# Write final merged file
[System.IO.File]::WriteAllBytes($outputPath, $merged)
Write-Host "Created $output (exactly 8MB)"
pause
exit
